<?php
namespace Aws\Credentials;

use Aws\Exception\CredentialsException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\PromiseInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * Credential provider that fetches container credentials with GET request.
 * container environment variables are used in constructing request URI.
 */
class EcsCredentialProvider
{
    const SERVER_URI = 'http://169.254.170.2';
    const ENV_URI = "AWS_CONTAINER_CREDENTIALS_RELATIVE_URI";
    const ENV_FULL_URI = "AWS_CONTAINER_CREDENTIALS_FULL_URI";
    const ENV_AUTH_TOKEN = "AWS_CONTAINER_AUTHORIZATION_TOKEN";
    const ENV_AUTH_TOKEN_FILE = "AWS_CONTAINER_AUTHORIZATION_TOKEN_FILE";
    const ENV_TIMEOUT = 'AWS_METADATA_SERVICE_TIMEOUT';
    const EKS_SERVER_HOST_IPV4 = '169.254.170.23';
    const EKS_SERVER_HOST_IPV6 = 'fd00:ec2::23';
    const ENV_RETRIES = 'AWS_METADATA_SERVICE_NUM_ATTEMPTS';

    const DEFAULT_ENV_TIMEOUT = 1.0;
    const DEFAULT_ENV_RETRIES = 3;

    /** @var callable */
    private $client;

    /** @var float|mixed */
    private $timeout;

    /** @var int */
    private $retries;

    /** @var int */
    private $attempts;

    /**
     *  The constructor accepts following options:
     *  - timeout: (optional) Connection timeout, in seconds, default 1.0
     *  - retries: Optional number of retries to be attempted, default 3.
     *  - client: An EcsClient to make request from
     *
     * @param array $config Configuration options
     */
    public function __construct(array $config = [])
    {
        $this->timeout = (float) isset($config['timeout'])
            ? $config['timeout']
            : (getenv(self::ENV_TIMEOUT) ?: self::DEFAULT_ENV_TIMEOUT);
        $this->retries = (int) isset($config['retries'])
            ? $config['retries']
            : ((int) getenv(self::ENV_RETRIES) ?: self::DEFAULT_ENV_RETRIES);

        $this->client = $config['client'] ?? \Aws\default_http_handler();
    }

    /**
     * Load container credentials.
     *
     * @return PromiseInterface
     * @throws GuzzleException
     */
    public function __invoke()
    {
        $this->attempts = 0;

        $uri = $this->getEcsUri();

        if ($this->isCompatibleUri($uri)) {
            return Promise\Coroutine::of(function () {
                $client = $this->client;
                $request = new Request('GET', $this->getEcsUri());

                $headers = $this->getHeadersForAuthToken();

                $credentials = null;

                while ($credentials === null) {
                    $credentials = (yield $client(
                        $request,
                        [
                            'timeout' => $this->timeout,
                            'proxy' => '',
                            'headers' => $headers,
                        ]
                    )->then(function (ResponseInterface $response) {
                        $result = $this->decodeResult((string)$response->getBody());
                        return new Credentials(
                            $result['AccessKeyId'],
                            $result['SecretAccessKey'],
                            $result['Token'],
                            strtotime($result['Expiration'])
                        );
                    })->otherwise(function ($reason) {
                        $reason = is_array($reason) ? $reason['exception'] : $reason;

                        $isRetryable = $reason instanceof ConnectException;
                        if ($isRetryable && ($this->attempts < $this->retries)) {
                            sleep((int)pow(1.2, $this->attempts));
                        } else {
                            $msg = $reason->getMessage();
                            throw new CredentialsException(
                                sprintf('Error retrieving credentials from container metadata after attempt %d/%d (%s)', $this->attempts, $this->retries, $msg)
                            );
             